require_relative "./escape.rb"
require_relative "./type.rb"
require_relative "./version.rb"

class Op < Struct.new(:name, :parse_nargs, :block, :ways)
  def dup
    Op.new(name.dup, parse_nargs, block, ways.map(&:dup))
  end
  def dup_mod(mod)
    return self if mod <= 0
    r = dup
    r.ways.each{|w| w.mod += mod if w.can_mod? }
    r
  end
end

class Way < Struct.new(:arg_ranks, :ret_rank, :arg_types, :ret_type, :mod, :vectorize, :promote, :impl)
  def mod_arg_ranks
    arg_ranks.map{|r| r < 0 ? ~r + mod : r }
  end
  def mod_ret_rank
    ret_rank < 0 ? ~ret_rank + mod : ret_rank
  end
  def dup
    Way.new(arg_ranks.dup, ret_rank, arg_types.dup, ret_type, mod, vectorize, promote, impl)
  end
  def can_mod?
    (arg_ranks + [ret_rank]).any?{|r| r < 0 }
  end
end

def lookup_op(name)
  (0..).each{|mod|
    op = Ops[name]
    return nil if op && mod>0 && !op.ways.any?(&:vectorize)
    return op.dup_mod(mod) if op
    return nil if !name[/,$/]
    name = $`
  }
end

def new_op(name, type_sig, block: false, vectorize: true, promote: true, &impl)
  promote = [promote] unless Array === promote
  parts = type_spec_parts(type_sig)
  *arg_ranks, ret_rank = parts.map{|arg| spec_rank(arg) }
  *arg_types, ret_type = parts.map{|arg| spec_type(arg) }

  way = Way.new(arg_ranks, ret_rank, arg_types, ret_type, 0, vectorize, promote, impl)
  Op.new(name, arg_ranks.size, block, [way])
end

Ops = {}
OpExamples = {}

if !defined? mk_op
  def mk_op(names, type_sig, example="", block: false, vectorize: true, promote: true, &impl)
    OpExamples[names] = example if example != ""
    names = names[/.[^(]*/].split

    op = new_op(names[-1], type_sig, block: block, vectorize: vectorize, promote: promote, &impl)

    names.each{|name|
      if Ops[name]
        Ops[name].ways << op.ways[0]
      else
        Ops[name] = op.dup
      end
    }

    if op.ways[0].can_mod? && op.ways[0].vectorize
      mod_op = op.dup
      mod_op.ways[0].mod = 1

      names.each{|name|
        if name =~ /^[a-z]/
          name = name.sub(/^./){|first|first.upcase}
          if Ops[name]
            Ops[name].ways << mod_op.ways[0]
          else
            Ops[name] = mod_op.dup
          end
        end
      }
    end
  end

  def category(name); end
  def quickref_op(names, desc, example)
    OpExamples[names] = example if example != ""
  end
end

NotFirst = :not_first
RepSnd = :rep_second

category("mathy ops")
mk_op("+ add", "int x -> x", '1 2+ -> 3', &add=->a,b{ a.value+b.value })
mk_op("+ add", "char int -> char", "'a 1+ -> 'b", &add)
mk_op("- sub", "x int -> x", '5 3- -> 2', &subt=->a,b{ a.value-b.value })
mk_op("- sub", "char char -> int", "'b 'a- -> 1", &subt)
mk_op("* mult", "int int -> int", '2 3* -> 6'){|a, b| a.value*b.value }
mk_op("/ div", "int int -> int", '7 2/ -> 3'){|a, b| b.value == 0 ? 0 : a.value/b.value }
mk_op("% mod", "x int -> int", '7 2% -> 1'){|a, b| b.value == 0 ? 0 : a.value%b.value }
mk_op("^ pow", "int int -> int", '2 3^ -> 8'){|a, b| pow(a.value, b.value) }
mk_op("~ negate", "int -> int", '1~ -> -1'){|a| -a.value }
mk_op("| abs", "int -> int", '5~| -> 5'){|a| a.value.abs }
mk_op("` str", "int -> [char]", '5` -> "5"'){|a| str(a) }
mk_op("( pred", "x -> x", '4( -> 3'){|a| a.value-1 }
mk_op(") succ", "x -> x", "'a) -> 'b"){|a| a.value+1 }
mk_op("} countTo", "int -> [int]", "3} -> [1,2,3]"){|a| range(1,a.value+1) }
mk_op("{ wholes", "[int]", "{ -> [0,1,2,..."){ range_from(0) }
mk_op("# toBase", "int int -> [int]", '6 2 # -> [1,1,0]'){|a,b| to_base(a.value,b.value) }
mk_op("#, fromBase", "[int] int -> int", '1,1,0 2 #, -> 6', promote: false){|a,b| from_base(a.value,b.value) }
mk_op("_ sum", "[x] -> int", '1,2,3,4 _ -> 10', promote: false){|a| to_strict_list(a).sum }

category("char ops")
mk_op("~ read", "[char] -> int", '"23"~ -> 23'){|a| read_num(a)[0] }
mk_op("} readAll", "[char] -> [int]", '"1,2 3"} -> [1,2,3]'){|a| split_non_digits(a) }
mk_op("| ord", "char -> int", '\'d| -> 100'){|a| a.value }
mk_op("` strip", "[char] -> [char]", '" ab\n"` -> "ab"'){|a| strip(a,$at,$rank) }
mk_op("* replicate", "[char] int -> [char]", '"ab"3* -> "ababab"'){|a, b|
  concat(take(b.value,repeat(a).const).const) }
mk_op("* join", "[[char]] [[char]] -> [char]", '1,2,3 " "* -> "1 2 3"', promote: RepSnd){|a, b| join(a,b)}
mk_op("/ split", "[char] [char] -> [[char]]", '"a  b c " " "/ -> ["a","b","c"]'){|a, b| s=cut(a,repeat(b).const).const; filter(s,s,nil,1) }
mk_op("% cut splitKeepEmpties", "[char] [[char]] -> [[char]]",'"a  b c " " "% -> ["a","","b","c",""]', promote: RepSnd){|a, b| cut(a,b) }

category("generic ops")
mk_op("a append", "[a] [a] -> [a]", '"ab" "cd"a -> "abcd"'){|a, b| append(a,b) }
mk_op("b backwards reverse", "[a] -> [a]", '"abc"b -> "cba"', promote: false){|a| reverse(a) }
mk_op("c cons", "[a] a -> [a]", '1,2 0c -> [0,1,2]'){|a, b| [b, a] }
mk_op("d drop", "[a] int -> [a]", '"abcde" 2d -> "cde"'){|a, b| drop(b.value, a) }
mk_op("g generate ncopies", "a int -> [a]", '4 3g -> [4,4,4]'){|a,b| take(b.value,repeat(a).const) }
mk_op("h head", "[a] -> a", '"abc"h -> \'a', promote: false){|a| z=zero($rank,$at); a.empty ? z : a.value[0].value }
mk_op("j just", "a -> [a]", "1j -> [1]"){|a|[a, [].const]}
mk_op("k keep take", "[a] int -> [a]", '"abcde" 2k -> "ab"'){|a, b| take(b.value, a) }
mk_op("l len", "[a] -> int", '"abc"l -> 3', promote: false){|a| len(a.value) }
mk_op("n not", "a -> int", '0,1,2n -> [1,0,0]', promote: false){|a| truthy($at, $rank, a.value) ? 0 : 1 }
mk_op("O Or o, or,", "int [char] -> [char]", promote: NotFirst){|a,b| truthy($at, 0, a.value) ? str(a) : b.value }
mk_op("o or", "a a -> a", '" bc" \'- o -> "-bc"', promote: NotFirst){|a,b| truthy($at, $rank, a.value) ? a.value : b.value }
mk_op("p nfoldProd", "[[a]] -> [[a]]", '"ab","12" p -> ["a1","a2","b1","b2"]', promote: false){|a| cart_prod(a) }
mk_op("q equal", "a a -> int", "1,2,3 1q -> [1,0,0]"){|a,b| spaceship(a, b, $rank) == 0 ? 1 : 0 }
mk_op("r repeat", "a -> [a]", "1r -> [1,1,1,1,1,1..."){|a| repeat(a) }
mk_op("s sortBy", "[a] [b] -> [a]", '"abc","d":len S -> ["d","abc"]', promote: false){|a,b| sort_by(a,b,$rank) }
mk_op("t tail", "[a] -> [a]", '"abc"t -> "bc"', promote: false){|a| a.empty ? [] : a.value[1].value }
mk_op("u unite concat", "[[a]] -> [a]", '"ab","cd","e"u -> "abcde"', promote: false){|a| concat(a) }
mk_op("v vet filter", "[a] [b] -> [a]", '"abcdef":2%v -> "ace"'){|a,b| filter(a,b,$bt,$rank) }
mk_op("w while takeWhile", "[a] [b] -> [a]", '"abc def":w -> "abc"', promote: false){|a,b| take_while(a,b,$bt,$rank) }
mk_op("x exclude setDiff", "[a] [a] -> [a]", '"abra" "ra"x -> "ba"'){|a,b| set_diff(a,b) }
mk_op("y and", "a b -> b", '" ab" "123"y -> " 23"', promote: NotFirst){|a,b| z=zero($rank,$bt); truthy($at, $rank, a.value) ? b.value : z }
mk_op("z zenith last", "[a] -> a", '"abc"z -> \'c', promote: false){|a| z=zero($rank,$at); last(a, z.const) }
mk_op("< lessThan", "a a -> int", '1,2,3 2< -> [1,0,0]'){|a,b| spaceship(a,b,$rank) == -1 ? 1 : 0 }
mk_op("\\ transpose", "[[a]] -> [[a]]", '"abc","def"\\ -> ["ad","be","cf"]', promote: false){|a| transpose(a) }
mk_op("& reshape", "[a] [int] -> [[a]]", '"abcdef" 2& -> ["ab","cd","ef"]', promote: RepSnd){|a,b| reshape(a,b) }
mk_op("? debut isFirst", "[a] -> [int]", '"aardvark"? -> [1,0,1,1,1,0,0,1]', promote: false){|a| isuniq(a) }
mk_op(". chunkWhen", "[a] [b] -> [[a]]", '"ab cd e":. -> ["ab ","cd ","e"]', promote: false){|a,b| chunk_while(a,b,$bt,$rank) }
mk_op("if", "a b b -> b", '0 1 2 if -> 2', promote: NotFirst){|a,b,c| truthy($at, $rank, a.value) ? b.value : c.value }
mk_op("nil", "[a]", 'nil -> []'){ [] }

category("folding ops") # (return type is type yielded)
mk_op("i iterate", "a -> [a]", '0 i 1,2,3+ -> [0,1,3,6]', block: true)
mk_op("e expand iterate0", "[a]", 'e 1,2,3+ -> [0,1,3,6]', block: true)
mk_op("f foldr", "a -> [a]", '0 f 1,2,3+ -> 6', block: true)
mk_op("m meld foldr0", "[a]", 'm 1,2,3+ -> 6', block: true)

category("stack ops")
mk_op(": dup", "a -> a", "5: -> 5 5")
mk_op("; mdup", "a -> a", "5;1- -> 4 5", block: true)

category("special symbols")
quickref_op("$", "arg", "5;$+ -> 10 5")
quickref_op("!", "pop from outer stack", "2 5;!+ -> 7 5")
quickref_op(">", "end block", "3;))> -> 5 3")
quickref_op(",", "unvec", '"ab"j, -> ["ab"]')
quickref_op("@", "register get/set", "@( 5@) -> 4 6")
quickref_op("[ ]", "push pop", "5[( ]) -> 4 6")
quickref_op("=", "assignment", "5=a( a) -> 4 6")

category("low rank overloads") # available W H and likely more
mk_op("B rangeFrom rangeBegin", "x -> [x]", "2B -> [2,3,4,..."){|a| range_from(a.value) }
mk_op("T rangeTo", "int -> [int]", "3T -> [0,1,2]"){|a| range(0,a.value) }
mk_op("H sqrt", "int -> int", "9H -> 3"){|a| square_root(a.value, 2)  }
mk_op("A min", "x x -> x", '4 5 A -> 4'){|a,b| spaceship(a,b,$rank) <= 0 ? a.value : b.value }
mk_op("X max", "x x -> x", '"bc" "ad" X -> "bd"'){|a,b| spaceship(a,b,$rank) >= 0 ? a.value : b.value }
mk_op("Z pow10", "int -> int", "2Z -> 100"){|a| pow(10, a.value) }
mk_op("P prod", "[int] -> int", '1,2,3,4 P -> 24', promote: false){|a| to_strict_list(a).inject(1,&:*) }
mk_op("\\ digits", "int -> [int]", '123 \\ -> [1,2,3]'){|a| v=to_base(a.value,10); v.empty? ? [0.const, Null] : v }
mk_op("U undigits", "[int] -> int", '1,2,3 U -> 123', promote: false){|a| undigits(a) }

category("debugging ops")
mk_op("stdin", "[char]", "stdin => <stdin>"){ ReadStdin.value }
mk_op("pad", "[a] a -> [a]", '1,2 3 pad -> [1,2,3,3,3...'){|a, b| padv(a, b) }
mk_op("show", "a -> [char]", '1,2 show -> "[1,2]"', vectorize: false, &show=->a{ inspectv($at, $rank, a) })
mk_op("type", "a -> [char]", '1,2 type -> "[int]"', vectorize: false){|a| str_to_lazy_list(type_and_rank_to_str($at, $rank)) }
mk_op("version", "[char]", "version -> \"#{Version}\""){ str_to_lazy_list(Version) }
mk_op("pp", "a -> [char]", '1,2 => prints [1,2]', vectorize: false, &show)
mk_op("nuke", "a -> a", '1,2 nuke -> ', vectorize: false){|a| raise}

category("internal ops")
mk_op("lines", "[char] -> [[char]]"){|a| lines(a) }
mk_op("superpad", "[a] -> [a]", promote: false){ raise }
mk_op("zero", "[] -> int"){|a| 0 } # default to int in fold0s
mk_op("zerom", "[a] -> a"){|a| zero($rank,$at) } # this is used for rank1+ fold0s
mk_op("zero", "[a] -> a"){|a| zero($rank,$at) }

# todo do this without duplicating logic/names
category("intuitive special promotes (don't need to show them in quickref)")
mk_op("* join", "[char] [[char]] -> [char]", promote: [RepSnd,NotFirst]){|a, b| join(map(a){|i| [i,Null] }.const,b)} # ideally don't promote first at all
mk_op("S SortBy sortBy, s,", "[[a]] [b] -> [[a]]", promote: false){|a,b| sort_by(a,b,$rank) }
mk_op("V Vet Filter vet, filter, v,", "[[a]] [b] -> [[a]]", promote: false){|a,b| filter(a,b,$bt,$rank) }
mk_op("W While TakeWhile w, while, takeWhile,", "[[a]] [b] -> [[a]]", promote: false){|a,b| take_while(a,b,$bt,$rank-1) }
mk_op("., ChunkWhen chunkWhen,", "[[a]] [b] -> [[[a]]]", promote: false){|a,b| chunk_while(a,b,$bt,$rank) }
mk_op("If if,", "[a] b b -> b", promote: NotFirst){|a,b,c| truthy($at, $rank+1, a.value) ? b.value : c.value }
